import { prisma } from './prisma';
import { startOfMonth } from 'date-fns';

export async function getCurrentUsage(subscriptionId: string) {
  const currentPeriod = startOfMonth(new Date());

  let usage = await prisma.usageCounter.findUnique({
    where: {
      subscriptionId_period: {
        subscriptionId,
        period: currentPeriod,
      },
    },
  });

  if (!usage) {
    usage = await prisma.usageCounter.create({
      data: {
        subscriptionId,
        period: currentPeriod,
      },
    });
  }

  return usage;
}

export async function incrementUsage(
  subscriptionId: string,
  type: 'products' | 'categories' | 'aiCredits',
  amount: number = 1
) {
  const currentPeriod = startOfMonth(new Date());

  const fieldMap = {
    products: 'productsUsed',
    categories: 'categoriesUsed',
    aiCredits: 'aiCreditsUsed',
  };

  await prisma.usageCounter.upsert({
    where: {
      subscriptionId_period: {
        subscriptionId,
        period: currentPeriod,
      },
    },
    create: {
      subscriptionId,
      period: currentPeriod,
      [fieldMap[type]]: amount,
    },
    update: {
      [fieldMap[type]]: {
        increment: amount,
      },
    },
  });
}

export async function decrementUsage(
  subscriptionId: string,
  type: 'products' | 'categories',
  amount: number = 1
) {
  const currentPeriod = startOfMonth(new Date());

  const fieldMap = {
    products: 'productsUsed',
    categories: 'categoriesUsed',
  };

  await prisma.usageCounter.updateMany({
    where: {
      subscriptionId,
      period: currentPeriod,
    },
    data: {
      [fieldMap[type]]: {
        decrement: amount,
      },
    },
  });
}

export async function getUsageStats(restaurantId: string) {
  const subscription = await prisma.subscription.findUnique({
    where: { restaurantId },
    include: {
      plan: true,
    },
  });

  if (!subscription) {
    throw new Error('Subscription not found');
  }

  const usage = await getCurrentUsage(subscription.id);

  const actualCounts = await prisma.restaurant.findUnique({
    where: { id: restaurantId },
    include: {
      _count: {
        select: {
          products: true,
          categories: true,
        },
      },
    },
  });

  return {
    plan: {
      name: subscription.plan.name,
      displayName: subscription.plan.displayName,
    },
    products: {
      used: actualCounts?._count.products || 0,
      limit: subscription.plan.maxProducts,
      percentage: Math.round(((actualCounts?._count.products || 0) / subscription.plan.maxProducts) * 100),
    },
    categories: {
      used: actualCounts?._count.categories || 0,
      limit: subscription.plan.maxCategories,
      percentage: Math.round(((actualCounts?._count.categories || 0) / subscription.plan.maxCategories) * 100),
    },
    aiCredits: {
      used: usage.aiCreditsUsed,
      limit: subscription.plan.aiCreditsMonthly,
      percentage: subscription.plan.aiCreditsMonthly > 0 
        ? Math.round((usage.aiCreditsUsed / subscription.plan.aiCreditsMonthly) * 100)
        : 0,
    },
    languages: {
      limit: subscription.plan.maxLanguages,
    },
    status: subscription.status,
    trialEnd: subscription.trialEnd,
    currentPeriodEnd: subscription.currentPeriodEnd,
    cancelAtPeriodEnd: subscription.cancelAtPeriodEnd,
    gracePeriodEnd: subscription.gracePeriodEnd,
  };
}