'use client';

import { useState } from 'react';
import { formatPrice } from '@/lib/utils';

interface Product {
  id: string;
  nameRo: string;
  nameEn: string | null;
  descriptionRo: string | null;
  descriptionEn: string | null;
  price: number;
  currency: string;
  imageUrl: string | null;
  allergens: string[] | null;
}

interface Category {
  id: string;
  nameRo: string;
  nameEn: string | null;
  icon: string | null;
  products: Product[];
}

interface Restaurant {
  name: string;
  logoUrl: string | null;
  categories: Category[];
}

interface PublicMenuProps {
  restaurant: Restaurant;
}

export default function PublicMenu({ restaurant }: PublicMenuProps) {
  const [language, setLanguage] = useState<'ro' | 'en'>('ro');
  const [search, setSearch] = useState('');

  const filteredCategories = restaurant.categories
    .map((cat) => ({
      ...cat,
      products: cat.products.filter((p) =>
        (language === 'ro' ? p.nameRo : p.nameEn || p.nameRo)
          .toLowerCase()
          .includes(search.toLowerCase())
      ),
    }))
    .filter((cat) => cat.products.length > 0);

  return (
    <div className="min-h-screen bg-gray-50">
      <header className="sticky top-0 z-10 bg-white border-b">
        <div className="container mx-auto px-4 py-4">
          <div className="flex items-center justify-between">
            {restaurant.logoUrl && (
              <img src={restaurant.logoUrl} alt={restaurant.name} className="h-12" />
            )}
            <h1 className="text-2xl font-bold">{restaurant.name}</h1>
            <button
              onClick={() => setLanguage(language === 'ro' ? 'en' : 'ro')}
              className="px-4 py-2 bg-gray-100 rounded-lg font-medium"
            >
              {language === 'ro' ? '🇬🇧 EN' : '🇷🇴 RO'}
            </button>
          </div>

          <input
            type="text"
            placeholder={language === 'ro' ? 'Caută...' : 'Search...'}
            value={search}
            onChange={(e) => setSearch(e.target.value)}
            className="w-full mt-4 px-4 py-2 border rounded-lg"
          />
        </div>
      </header>

      <div className="container mx-auto px-4 py-6">
        {filteredCategories.map((category) => (
          <div key={category.id} className="mb-8">
            <h2 className="text-xl font-bold mb-4 flex items-center gap-2">
              {category.icon && <span>{category.icon}</span>}
              {language === 'ro' ? category.nameRo : category.nameEn || category.nameRo}
            </h2>

            <div className="grid gap-4 md:grid-cols-2 lg:grid-cols-3">
              {category.products.map((product) => (
                <div key={product.id} className="bg-white rounded-lg shadow p-4">
                  {product.imageUrl && (
                    <img
                      src={product.imageUrl}
                      alt={product.nameRo}
                      className="w-full h-48 object-cover rounded-lg mb-3"
                    />
                  )}

                  <h3 className="font-bold text-lg">
                    {language === 'ro' ? product.nameRo : product.nameEn || product.nameRo}
                  </h3>

                  {(product.descriptionRo || product.descriptionEn) && (
                    <p className="text-sm text-gray-600 mt-1">
                      {language === 'ro'
                        ? product.descriptionRo
                        : product.descriptionEn || product.descriptionRo}
                    </p>
                  )}

                  <div className="mt-3 flex justify-between items-center">
                    <span className="text-lg font-bold text-green-600">
                      {formatPrice(Number(product.price), product.currency)}
                    </span>

                    {product.allergens && product.allergens.length > 0 && (
                      <div className="flex gap-1">
                        {product.allergens.map((allergen) => (
                          <span
                            key={allergen}
                            className="text-xs px-2 py-1 bg-yellow-100 rounded"
                            title={allergen}
                          >
                            ⚠️
                          </span>
                        ))}
                      </div>
                    )}
                  </div>
                </div>
              ))}
            </div>
          </div>
        ))}

        {filteredCategories.length === 0 && (
          <div className="text-center text-gray-500 py-12">
            {language === 'ro' ? 'Niciun produs găsit' : 'No products found'}
          </div>
        )}
      </div>

      <footer className="bg-white border-t py-4 text-center text-sm text-gray-500">
        Powered by <strong>Vestho Digital Menu</strong>
      </footer>
    </div>
  );
}