import { NextRequest, NextResponse } from 'next/server';
import { getServerSession } from 'next-auth';
import { authOptions } from '@/lib/auth';
import { prisma } from '@/lib/prisma';
import { categorySchema } from '@/lib/validations';
import { enforceLimit, SubscriptionError } from '@/lib/subscription-guard';
import { incrementUsage } from '@/lib/usage';

export async function GET(req: NextRequest) {
  try {
    const session = await getServerSession(authOptions);
    
    if (!session?.user) {
      return NextResponse.json({ error: 'Neautentificat' }, { status: 401 });
    }

    const restaurant = await prisma.restaurant.findFirst({
      where: { ownerId: session.user.id },
    });

    if (!restaurant) {
      return NextResponse.json({ error: 'Restaurant negăsit' }, { status: 404 });
    }

    const categories = await prisma.category.findMany({
      where: { restaurantId: restaurant.id },
      orderBy: { sortOrder: 'asc' },
      include: {
        _count: {
          select: { products: true },
        },
      },
    });

    return NextResponse.json(categories);
  } catch (error) {
    console.error('Get categories error:', error);
    return NextResponse.json({ error: 'Eroare server' }, { status: 500 });
  }
}

export async function POST(req: NextRequest) {
  try {
    const session = await getServerSession(authOptions);
    
    if (!session?.user) {
      return NextResponse.json({ error: 'Neautentificat' }, { status: 401 });
    }

    const restaurant = await prisma.restaurant.findFirst({
      where: { ownerId: session.user.id },
      include: { subscription: true },
    });

    if (!restaurant) {
      return NextResponse.json({ error: 'Restaurant negăsit' }, { status: 404 });
    }

    await enforceLimit(restaurant.id, 'categories');

    const body = await req.json();
    const data = categorySchema.parse(body);

    const maxSortOrder = await prisma.category.findFirst({
      where: { restaurantId: restaurant.id },
      orderBy: { sortOrder: 'desc' },
      select: { sortOrder: true },
    });

    const category = await prisma.category.create({
      data: {
        restaurantId: restaurant.id,
        ...data,
        sortOrder: (maxSortOrder?.sortOrder || 0) + 1,
      },
    });

    if (restaurant.subscription) {
      await incrementUsage(restaurant.subscription.id, 'categories', 1);
    }

    await prisma.auditLog.create({
      data: {
        userId: session.user.id,
        action: 'CREATE',
        entity: 'category',
        entityId: category.id,
        metadata: { name: category.nameRo },
      },
    });

    return NextResponse.json(category);
  } catch (error: any) {
    if (error instanceof SubscriptionError) {
      return NextResponse.json(
        { error: error.message, code: error.code },
        { status: 403 }
      );
    }
    console.error('Create category error:', error);
    return NextResponse.json({ error: 'Eroare server' }, { status: 500 });
  }
}